//___________________________________
//                                   \
// DynaLayer, Dynamic Layer swapping, extending Lithonite.  version 1.1 $Id: dynalayer.js 64 2007-09-09 06:51:17Z nilton $
//___________________________________/


/**
 * Extend LithoniteEngine with a function to dynamically move a tile down a layer, to be able to walk around
 * an object and not worry about proper layering.
 * Requires Lithonite and Phenibut
 * Coded By FutureBoyNil based on an idea of Radnen (Andrew Helenius)
 *
 * To create the illusion that a sprite can walk in front and behind a object created with obstructed tiles, the base tile is on the base layer,
 * and the tiles that represent something in the air are on a layer above. When your sprite must actually be able to walk in front and behind something and
 * you cant make things to look right, enter DynaLayer.
 * 
 * Just after a mapchange (in the map entry script, for example), call DynaCheck()
 * If the name layer where your sprite is walking on starts with "_" then DynaLayer is activated.
 * It will move all adjacent tiles north of the sprite that are one layer up, to the layer the sprite
 * is standing on. This will happen only if for that tile the tileindex on the sprite's layer is 0.
 * Make your first tile transparant. And dont forget to put a real tile one layer down.
 * You also need to put DynaLayer() inside your update script for it to work.
 *
 * 1) In your initialization scripts (called only once). Without this Sphere dies with:"this.DLA has no properties"
 *   Lithonite.DynaLayerInit(2); // See the function DynaLayerInit for more information
 * 2) In your update script 
 *   Lithonite.DynaLayer();
 *
 * Before each mapchange, call DynaClear(). After each mapchange, call DynaCheck();
 *
 * Before each layer change of your sprite, call DynaLift(), and after call DynaCheck();
 * For this layer change to run automatically, redefine SetPerson Layer, like so:
 *
 * // Copy function SetPersonLayer to SetPersonLayer_
 * if(typeof SetPersonLayer_ == 'undefined') SetPersonLayer_ = SetPersonLayer;
 *
 * // Overload SetPersonLayer with DynaLayer checking code:
 * SetPersonLayer = function(name, layer){
 *   if(name != Lithonite.GIP) return SetPersonLayer_(name, layer);
 *   Lithonite.DynaLift();
 *   var ret = SetPersonLayer_(name, layer);
 *   Lithonite.DynaCheck();
 *   return ret;
 * }
 *
 * Bugs: Does not work for NPC's. Your sprite must not be wider than 3 tiles (and the sprite base must be centered)
 * Hacks: Its not hard to remove the dependencies on Lithonite. Replace GIP with the name of your sprite 
 * and MapToTile(X/Y) with: MapToTileX = function(a){return a>>4};
 */

/**
 * This function has to be put in your update script for Dynamic Layering to work.
 */
LithoniteEngine.prototype.DynaLayer = function() {};

LithoniteEngine.prototype.DynaLayerActive = false; // Will tell you if DynaLayer is active or not.

LithoniteEngine.prototype.GL = 0; // GIP layer
LithoniteEngine.prototype.GTX = 0; // GIP Tile X
LithoniteEngine.prototype.GTY = 0; // GIP Tile Y
LithoniteEngine.prototype.DLH = 2; // DynaLayer height, 2 tiles lookup by default.
LithoniteEngine.prototype.GLH2 = 0; // Get Map Layer height - 2
LithoniteEngine.prototype.GLW1 = 0; // Get Map Layer Width - 1

// Overload SetPersonLayer with DynaLayer checking code:
if(typeof SetPersonLayer_ == 'undefined')
	SetPersonLayer_ = SetPersonLayer;
SetPersonLayer = function(name, layer){
	if(name != Game.lithonite.GIP) return SetPersonLayer_(name, layer);
	Game.lithonite.DynaLift();
	SetPersonLayer_(name, layer);
	Game.lithonite.DynaCheck();
}

/**
 * This creates the DynaLayerArray, you need to call this once at the beginning of your game,
 * and before each mapchange, so DynaLift doesnt try to lift tiles from the old map.
 * Depending on the height and base of your sprite, you need to lookup 1, 2 or 3 tiles.
 * So Base=16 pixels, height of spriteset is 32: lookup = 1. If base is smaller, use a value of 2.
 * @param {integer} vlookup Height in tiles for Dynamic Layering. Defaults to 2.
 * example: Lithonite.DynaLayerInit();
 */
LithoniteEngine.prototype.DynaLayerInit = function(vlookup) {
	if(vlookup) this.DLH = vlookup;
	this.DLA = new Array(this.DLH);
	for(var i=0,l=this.DLH;i<l;++i){ this.DLA[i]=new Array(3); }
};


/**
 * Clears all Flattening info. Call before we change maps
 * @param {boolean} force Force a clear, even if DynaLayer is not active.
 */
LithoniteEngine.prototype.DynaClear = function(force) {
	if(!force && !this.DynaLayerActive) return;
	var j=this.DLH-1,i;
	do{
		i = 2;
		do{ this.DLA[j][i] = undefined; }while(i-->0);
	}while(j-->0);
	this.DynaLayer=function(){};
}


/**
 * Undo the Layer Flattening. Must be on the same map.
 * // SetTile.apply(null,t); // Slower due to @&$#&^! stitch function!
 * // SetTile(t[0],t[1],t[2],t[3]); // Had to use this one instead.
 */
LithoniteEngine.prototype.DynaLift = function() {
	var t;
	var j=this.DLH-1,i;
	do{
		i = 2;
		do{
			if(t = this.DLA[j][i]){
				SetTile(t[0], t[1], t[2]+1, GetTile(t[0],t[1],t[2]));
 				SetTile.apply(null,t);
				this.DLA[j][i] = undefined;
			}
		}while(i-->0);
	}while(j-->0);
}

/**
 * Internal function. Flatten a layer.
 * It stores the information in a matrix. Each flattened element is an array.
 * This array contains the following information: x, y, layer and replaced tile.
 * MAYBE: add GetTileName support
 */
LithoniteEngine.prototype.DynaFlatten = function() {
	var xx, yy;
	for(var j=1,l=this.DLH+1;j<l;++j){
		 for(var i=2;i>=0;--i){
			xx = this.GTX+i-1;
			yy = this.GTY-j;
			if(xx<0 || xx>this.GLW1) continue;
			if(yy<0 || yy>this.GLH2) continue;
			if(!GetTile(xx, yy, this.GL) && GetTile(xx, yy+1, this.GL)){
				this.DLA[j-1][i] = [xx,yy,this.GL,GetTile(xx,yy,this.GL)];
				SetTile(xx, yy, this.GL, GetTile(xx, yy, this.GL+1));
				SetTile(xx, yy, this.GL+1, 0);
			}
		 }
	}
}

/**
 * When the layer of the input person changes, call this function, it will DynaLift, DynaCheck 
 * (and DynaFlatten, if required)
 * @returns true if DynaLayer is active, false if not.
 */
LithoniteEngine.prototype.DynaLayerUpdate = function() {
	if(this.GL != GetPersonLayer(this.GIP)){
		this.DynaLift();
		return this.DynaCheck();
	};
	return this.DynaLayerActive;
};

/**
 * Call after each mapchange and SetPersonLayer to check if Dynamic Layering has to be turned on or off,
 * You need to have called DynaLayerInit(), DynaLift() or DynaClear() first before using this function.
 * @returns true if its a dynamic layer (DynaLayer() is automatically activated), false if not.
 */
LithoniteEngine.prototype.DynaCheck = function(){
	if( GetLayerName(GetPersonLayer(this.GIP))[0] != "_" ){
		this.DynaLayer = function(){};
		return this.DynaLayerActive = false;
	}
	this.GLW1 = GetLayerWidth(GetPersonLayer(this.GIP))-1;
	this.GLH2 = GetLayerHeight(GetPersonLayer(this.GIP))-2;

	this.GL = GetPersonLayer(this.GIP);
	this.GTX = this.MapToTileX(GetPersonX(this.GIP));
	this.GTY = this.MapToTileY(GetPersonY(this.GIP));
	this.GTY2 = this.GTY - 2 - this.DLH;

	this.DynaLayer = function(){ 
		if(this.idle)return;
		this.dlnc = 1;
		if(this.moving){
			var T = this.MapToTileX(GetPersonX(this.GIP));
			if(T != this.GTX){
				this.dlnc = 0;
				this.GTX = T;
			}
			T = this.MapToTileY(GetPersonY(this.GIP));
			if(T != this.GTY){
				this.dlnc = 0;
				this.GTY = T;
				this.GTY2 = this.GTY - 2 - this.DLH;
			}
		}
		if(this.dlnc) return;
		this.GL = GetPersonLayer(this.GIP);
		this.DynaLift();
		this.DynaFlatten();
	};

	this.DynaFlatten();
	GarbageCollect();
	return this.DynaLayerActive = true;
}
